// This project is released under the GPL Licence

// ToDo: dB, hz or ms values in the readout. sanitylimits on sr? set defaults to buffer 64/200, release 0.5090.
// Known issues: There is some roundoff error in the optimized filter, causing gainoffsets. 
// Altfilter option? Maybe I could squeeze another point into the filter also, by truncating the beginning.
/*****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>

/*****************************************************************************/

#include "ladspa.h"

/*****************************************************************************/

/* The port numbers for the plugin: */

#define PXU_CONTROL1 0
#define PXU_CONTROL2 1
#define PXU_CONTROL3 2

#define PXU_INPUT1  3
#define PXU_OUTPUT1 4
#define PXU_INPUT2  5
#define PXU_OUTPUT2 6

#define PXU_LATENCY 7
/*****************************************************************************/
	
	
/* The structure used to hold port connection information and state
   (actually gain controls require no further state). */

typedef struct {

  /* Ports:
     ------ */
  LADSPA_Data * m_pfControlValue1;
  LADSPA_Data * m_pfControlValue2;
  LADSPA_Data * m_pfControlValue3;
  LADSPA_Data * m_pfInputBuffer1;
  LADSPA_Data * m_pfOutputBuffer1;
  LADSPA_Data * m_pfInputBuffer2;  /* (Not used for mono) */
  LADSPA_Data * m_pfOutputBuffer2; /* (Not used for mono) */

  LADSPA_Data * m_pfLatency;

  double b_sr;
  double b_sensitivity;
  int i_cnt;
  int i_bfcn;
  int i_optcnt;
  int i_h1sortcurr;
  double b_h1highest;
  int i_h2sortcurr;		
  double b_h2highest;
  double b_lenv;
  double b_cmpslp;
  double b_cmpcut;
  double b_cmpla;
  double b_cmprel;
  double b_h1buf[1002];
  double b_h2buf[1002];
  double b_avg1buf[1002];
  double b_avg2buf[1002];
  double b_avg3buf[1002];
  double b_avg4buf[1002];
  double b_avg5buf[1002];
  double b_syncbufL[1002];
  double b_syncbufR[1002];
  double b_syncbufrr[1002];
  double b_nr1;
  double b_nr2;
  double b_nr3;
  double b_nr4;
  double b_nr5;
  double b_at1;
  double b_at2;
  double b_at3;
  double b_at4;
  double b_at5;
  double b_tg1;
  double b_tg2;
  double b_tg3;
  double b_tg4;
  double b_tg5;
  double b_tg6;
  double b_tg7;
  double b_tg8;
  double b_tg9;
  double b_tg10;
  double b_tg11;
  double b_tg12;
  double b_tg13;
  double b_tg14;
  double b_tg15;
  double b_tg16;
  double b_tg17;
  double b_tg18;
  double b_tg19;
  double b_tg20;
  double b_tg21;
  double b_tg21n;
  double b_gse1buf;
  double b_gse2buf;
  double b_gse3buf;
  double b_gse4buf;
  double b_gse5buf;
  
} Pxu;


/*****************************************************************************/


LADSPA_Handle 
instantiatePxu(const LADSPA_Descriptor * Descriptor,
			   unsigned long             SampleRate) {

  Pxu * psPxu;

  psPxu 
    = (Pxu *)malloc(sizeof(Pxu));

  if (psPxu == NULL) 
    return NULL;
  
    psPxu->b_sr = (LADSPA_Data)SampleRate;
    psPxu->i_cnt = 0;
    psPxu->i_bfcn = 0;
    psPxu->i_optcnt = 10001; //initialize avgbufs
    psPxu->i_h1sortcurr = 0;
    psPxu->b_h1highest = 0;
    psPxu->i_h2sortcurr = 0;		
    psPxu->b_h2highest = 1;
    psPxu->b_lenv = 1;
    psPxu->b_cmpslp = -1;
    psPxu->b_cmpcut = -1;
    psPxu->b_cmpla = -1;
    psPxu->b_cmprel = -1;
    psPxu->b_gse1buf = 1;
    psPxu->b_gse2buf = 1;
    psPxu->b_gse3buf = 1;
    psPxu->b_gse4buf = 1;
    psPxu->b_gse5buf = 1; 
//    psPxu->b_sensitivity = pow(2,(((0.8570)*-1000))/ 6); // denormals problem.
    psPxu->b_sensitivity = pow(2,(((0.0200)*-1000))/ 6);
    
    int i_cnt = 0;
    while (i_cnt < 1002) {
      psPxu->b_syncbufrr[i_cnt] = psPxu->b_sensitivity;
      psPxu->b_h1buf[i_cnt] = psPxu->b_sensitivity;
      psPxu->b_h2buf[i_cnt] = 1;
      psPxu->b_syncbufL[i_cnt] = 0;
      psPxu->b_syncbufR[i_cnt] = 0;
      psPxu->b_avg1buf[i_cnt] = 1;
      psPxu->b_avg2buf[i_cnt] = 1;
      psPxu->b_avg3buf[i_cnt] = 1;
      psPxu->b_avg4buf[i_cnt] = 1;
      psPxu->b_avg5buf[i_cnt] = psPxu->b_sensitivity;
      i_cnt++;
    }
  
  return psPxu;
}


/*****************************************************************************/

/* Connect a port to a data location. */
void 
connectPortToPxu(LADSPA_Handle Instance,
		       unsigned long Port,
		       LADSPA_Data * DataLocation) {

  Pxu * psPxu;

  psPxu = (Pxu *)Instance;
  switch (Port) {
  case PXU_CONTROL1:
    psPxu->m_pfControlValue1 = DataLocation;  
    break;
  case PXU_CONTROL2:
    psPxu->m_pfControlValue2 = DataLocation; 
    break;
  case PXU_CONTROL3:
    psPxu->m_pfControlValue3 = DataLocation; 
    break;
  case PXU_INPUT1:
    psPxu->m_pfInputBuffer1 = DataLocation;
    break;
  case PXU_OUTPUT1:
    psPxu->m_pfOutputBuffer1 = DataLocation;
    break;
  case PXU_INPUT2:
    /* (This should only happen for stereo.) */
    psPxu->m_pfInputBuffer2 = DataLocation;
    break;
  case PXU_OUTPUT2:
    /* (This should only happen for stereo.) */
    psPxu->m_pfOutputBuffer2 = DataLocation;
    break;
  case PXU_LATENCY:
    psPxu->m_pfLatency = DataLocation;  
    break;
  }

}

/*****************************************************************************/

void 
runMonoPxu(LADSPA_Handle Instance,
		 unsigned long SampleCount) {
  
  LADSPA_Data * pfInput;
  LADSPA_Data * pfOutput;
  LADSPA_Data fVar1;
  LADSPA_Data fVar2;
  LADSPA_Data fVar3;
  Pxu * psPxu;
  unsigned long lSampleIndex;

  psPxu = (Pxu *)Instance;

  pfInput = psPxu->m_pfInputBuffer1;
  pfOutput = psPxu->m_pfOutputBuffer1;
  fVar1 = *(psPxu->m_pfControlValue1);
  fVar2 = *(psPxu->m_pfControlValue2);
  fVar3 = *(psPxu->m_pfControlValue3);

  for (lSampleIndex = 0; lSampleIndex < SampleCount; lSampleIndex++) 
    *(pfOutput++) = *(pfInput++) * fVar1;
}

/*****************************************************************************/

void 
runStereoPxu(LADSPA_Handle Instance,
		   unsigned long SampleCount) {
  
  LADSPA_Data * pfInput1;
  LADSPA_Data * pfInput2;
  LADSPA_Data * pfOutput1;
  LADSPA_Data * pfOutput2;
  LADSPA_Data fVar1;
  LADSPA_Data fVar2;
  LADSPA_Data fVar3;
	  LADSPA_Data * pfLatency;
  
  Pxu * psPxu;
  unsigned long lSampleIndex;

  psPxu = (Pxu *)Instance;
  
  fVar1 = *(psPxu->m_pfControlValue1);
  fVar2 = *(psPxu->m_pfControlValue2);
  fVar3 = *(psPxu->m_pfControlValue3);
  
  pfInput1 = psPxu->m_pfInputBuffer1;
  pfInput2 = psPxu->m_pfInputBuffer2;
  pfOutput1 = psPxu->m_pfOutputBuffer1;
  pfOutput2 = psPxu->m_pfOutputBuffer2;
  pfLatency = psPxu->m_pfLatency;
  
	double b_of1,b_of2,b_of3,b_of4,b_of5;
	double b_cErr;
	double b_cSum;
	double b_cSumpre;
	double b_cSub;
	double b_cIn;

  	int i_optimize = 10001;

  	// adjust to samplerate 
  	double b_sr = psPxu->b_sr;                   
  	double b_srt = b_sr/44100;

  	// front
  	double b_ingain = pow(2,(((fVar1-0.5)*100))/ 6); 
  
 	double b_la = fVar2 * 200; if (b_la < 20) {b_la = 20;}; b_la = (b_la * b_srt) + 0.5; int i_la = b_la; 
  	if (b_la != psPxu->b_cmpla) {psPxu->i_optcnt = i_optimize; psPxu->b_cmpla = b_la;}
  	int i_htravlensub = i_la;
  	int i_htravlen = i_htravlensub + 1;
  
  	double b_release = (pow(2,(((1-(fVar3))*-100))/ 6)) / b_srt;  
 
  	// internal
  	int i_bufpoint;
  	double b_sub;
  	psPxu->i_cnt = 0;

  	int i_avglen = i_htravlen*3; // 3x OS in filter, reduces noise, and improves accuracy.
  	int i_avglen2 = ((i_la * 2) + 1); 
  	double b_htravlen = i_avglen;
	i_avglen = i_htravlen; // tmp
//int i_avglen3 = fvarg8 * 100;
//double b_totlen = i_avglen3; 
//b_totlen = b_totlen - 1;
//b_totlen = b_totlen / 2;
//int i_totdel = b_totlen * 4;
//int i_totdel0 = i_totdel;

	// tweaked for "quiet" psychoacoustic.
	double b_tgcut = 0.1640;
	double b_tgslp = (0.9830*0.0100)+0.9900;

	b_tgcut = 1-(b_tgcut/b_htravlen*50);
	if (b_tgcut != psPxu->b_cmpcut) {psPxu->i_optcnt = i_optimize; psPxu->b_cmpcut = b_tgcut;}
	if (b_tgslp != psPxu->b_cmpslp) {psPxu->i_optcnt = i_optimize; psPxu->b_cmpslp = b_tgslp;}
	double b_tgcut1 = b_tgcut * b_tgslp * b_tgslp * b_tgslp * b_tgslp;
	double b_tgcut2 = b_tgcut * b_tgslp * b_tgslp * b_tgslp;
	double b_tgcut3 = b_tgcut * b_tgslp * b_tgslp;
	double b_tgcut4 = b_tgcut * b_tgslp;
	double b_tgcut5 = b_tgcut;
	b_tgcut1 = 1-b_tgcut1;
	b_tgcut2 = 1-b_tgcut2;
	b_tgcut3 = 1-b_tgcut3;
	b_tgcut4 = 1-b_tgcut4;
	b_tgcut5 = 1-b_tgcut5;
	double b_tggain1,b_tggain2,b_tggain3,b_tggain4,b_tggain5;
	b_tggain1 = 1;
	b_tggain2 = -4;
	b_tggain3 = 6;
	b_tggain4 = -4;
	b_tggain5 = 1;
	b_tggain1 = b_tggain1 / b_tgcut1;
	b_tggain2 = b_tggain2 / b_tgcut2;
	b_tggain3 = b_tggain3 / b_tgcut3;
	b_tggain4 = b_tggain4 / b_tgcut4;
	b_tggain5 = b_tggain5 / b_tgcut5;
	
//	double b_tgnrm = 1/(b_tggain1 + b_tggain2 + b_tggain3 + b_tggain4 + b_tggain5);
	b_of1 = b_tggain1;
	b_of2 = b_tggain2;
	b_of3 = b_tggain3;
	b_of4 = b_tggain4;
	b_of5 = b_tggain5;

	b_cErr = 0;
	b_cSum = 0;

	b_cIn = b_of1;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
	b_cIn = b_of2;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of3;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of4;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of5;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;
	
	double b_tgnrm = 1/b_cSum;
	
	b_tggain1 = b_tggain1 * b_tgnrm;
	b_tggain2 = b_tggain2 * b_tgnrm;
	b_tggain3 = b_tggain3 * b_tgnrm;
	b_tggain4 = b_tggain4 * b_tgnrm;
	b_tggain5 = b_tggain5 * b_tgnrm;

// Fast Gaussian filter (variable 1th to 9nth order lowpass, minimal phase)
	double b_gse1gain;
	double b_gse2gain;
	double b_gse3gain;
	double b_gse4gain;
	double b_gse5gain;

	b_gse1gain = 1;
	b_gse2gain = -4;
	b_gse3gain = 6;
	b_gse4gain = -4;
	b_gse5gain = 1;

	double b_gse1cut;
	double b_gse2cut;
	double b_gse3cut;
	double b_gse4cut;
	double b_gse5cut;

	// max saturation of transient, more is unnatural, less is unessecary.
	double b_gseslope = 1-pow(2,(((0.5770)*-100))/ 6);  // 0.5 = stable.. 
	double b_gsecut = 1-pow(0.4220,6); //
	b_gsecut = b_gsecut / b_srt; // adjust to samplerate.

	b_gse1cut = b_gsecut * b_gseslope * b_gseslope * b_gseslope * b_gseslope;
	b_gse2cut = b_gsecut * b_gseslope * b_gseslope * b_gseslope;
	b_gse3cut = b_gsecut * b_gseslope * b_gseslope;
	b_gse4cut = b_gsecut * b_gseslope;
	b_gse5cut = b_gsecut;

	b_gse1cut = 1-b_gse1cut;
	b_gse2cut = 1-b_gse2cut;
	b_gse3cut = 1-b_gse3cut;
	b_gse4cut = 1-b_gse4cut;
	b_gse5cut = 1-b_gse5cut;

	b_gse1gain = b_gse1gain / b_gse1cut;
	b_gse2gain = b_gse2gain / b_gse2cut;
	b_gse3gain = b_gse3gain / b_gse3cut;
	b_gse4gain = b_gse4gain / b_gse4cut;
	b_gse5gain = b_gse5gain / b_gse5cut;

	b_of1 = b_gse1gain;
	b_of2 = b_gse2gain;
	b_of3 = b_gse3gain;
	b_of4 = b_gse4gain;
	b_of5 = b_gse5gain;

	b_cErr = 0;
	b_cSum = 0;

	b_cIn = b_of1;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
	b_cIn = b_of2;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of3;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of4;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of5;
	b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
	b_cSum = b_cSum + b_cSub;
	b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
	double b_gsenrm = 1/b_cSum;

	b_gse1gain = b_gse1gain * b_gsenrm;
	b_gse2gain = b_gse2gain * b_gsenrm;
	b_gse3gain = b_gse3gain * b_gsenrm;
	b_gse4gain = b_gse4gain * b_gsenrm;
	b_gse5gain = b_gse5gain * b_gsenrm;

//
//////////

//double b_spow1 = (fvarg16*10000) + 1;
//double b_spow2 = 1/b_spow1;
//b_spow1 = 1/b_spow1; b_spow2 = 1/b_spow2;
//double b_spoff = pow(2,(((fvarg15-0.5)*200))/ 6); 
//
//double b_spow3 = (fvarg18*10) + 1;
//double b_spoff2 = pow(2,(((fvarg17-0.5)*2000))/ 6); 



//	double b_rtype = 0.805;
	double b_discam = 0.3600*0.01;

	double b_f1;
	double b_f2;
	double b_f3;
	double b_f4;
		
		
  for (lSampleIndex = 0; lSampleIndex < SampleCount; lSampleIndex++) 
 {
   double b_in1 = *(pfInput1++); 
   double b_in2 = *(pfInput2++); 

		b_in1 = b_in1 * b_ingain;
		b_in2 = b_in2 * b_ingain;
		
		int i_syncread = psPxu->i_bfcn - i_la; 
		if (i_syncread < 0) {i_syncread = 1002 + i_syncread;}

		// Rectification and SH for envelope and detector.
		double b_inrr;
		int i_inrr2 = 0;
		double b_inr = b_in1; if (b_inr < 0) {b_inr = -b_inr;}
		double b_inr2 = b_in2; if (b_inr2 < 0) {b_inr2 = -b_inr2;}
		if (b_inr2 > b_inr) {b_inr = b_inr2; i_inrr2 = 1;}

		b_inrr = b_inr;
		if (i_inrr2 == 0) {if (b_in1 < 0) {b_inrr = -b_inrr;}}
		else if (i_inrr2 == 1) {if (b_in2 < 0) {b_inrr = -b_inrr;}}
		psPxu->b_syncbufrr[psPxu->i_bfcn] = b_inrr;
//		double b_inrrsync = psPxu->b_syncbufrr[i_syncread];

		if (b_inr < psPxu->b_sensitivity) {b_inr = psPxu->b_sensitivity;} // sensitivity and zero removal..

		double b_inrsh = b_inr;
		psPxu->b_h1buf[psPxu->i_bfcn] = b_inrsh;
		if (psPxu->b_h1highest < b_inrsh) {
			psPxu->i_h1sortcurr = 0;
			psPxu->b_h1highest = b_inrsh;
		}
		else if (psPxu->i_h1sortcurr > i_htravlensub) {
			int cnt = 0;
			psPxu->b_h1highest = 0;
			while (cnt < i_htravlen) {
				int i_bufpoint = psPxu->i_bfcn - cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				if (psPxu->b_h1highest < psPxu->b_h1buf[i_bufpoint]) {
					psPxu->b_h1highest = psPxu->b_h1buf[i_bufpoint];
					psPxu->i_h1sortcurr = cnt;
				}
				cnt++;
			}
		}
		psPxu->i_h1sortcurr++;
		if (psPxu->i_h1sortcurr > 1002) {psPxu->i_h1sortcurr = 1002;}
		b_inrsh = psPxu->b_h1highest;
		double b_inrsync = psPxu->b_h1buf[i_syncread];

/////////////////////////////////////////////////////////////////////////////
		// bandpass, 200hz-4.5khz, for midrange sensitive envelope movement.
		// not implemented in v1.0.

		double b_rms = b_inr ;
		b_rms = b_rms*b_rms;
		double b_f5 = b_rms;

		psPxu->b_avg5buf[psPxu->i_bfcn] = b_f5; //

		i_bufpoint = psPxu->i_bfcn - i_avglen2;
		if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}

		b_sub = psPxu->b_avg5buf[i_bufpoint];
		psPxu->b_tg21 = psPxu->b_tg21 + b_f5 - b_sub;

		if (psPxu->i_optcnt == i_optimize) {
			psPxu->b_tg21n = i_avglen2;
			psPxu->b_tg21n = 1/psPxu->b_tg21n;
		// make filter correct with regards to previous values
			psPxu->i_cnt = 0;
			psPxu->b_tg21 = 0;
			while (psPxu->i_cnt < i_avglen2) {
				i_bufpoint = psPxu->i_bfcn - psPxu->i_cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				psPxu->b_tg21 = psPxu->b_tg21 + psPxu->b_avg5buf[i_bufpoint];
				psPxu->i_cnt++;
			}

		}
		b_f5 = psPxu->b_tg21 * psPxu->b_tg21n;
		b_rms = sqrt(b_f5);
	if (b_rms < 0) {b_rms = 0;}

//.. not implemented, mid-range sensitive envelope movement.
//		double b_lowf = b_f6;
//
//		double b_prefer = b_lowf + ((-b_lowf + b_inrr45) * fvarg11);
//		double b_region = b_inrrsync - (b_lowf + b_inrr45);
//		if (b_prefer < 0) {b_prefer = -b_prefer;}
//		if (b_prefer < psPxu->b_sensitivity) {b_prefer = psPxu->b_sensitivity;}
//		if (b_region < 0) {b_region = -b_region;}
//		if (b_region < psPxu->b_sensitivity) {b_region = psPxu->b_sensitivity;}
//
//		double b_lowfr = b_lowf;
//		if (b_lowfr < 0) {b_lowfr = -b_lowfr;}
//		if (b_lowfr < psPxu->b_sensitivity) {b_lowfr = psPxu->b_sensitivity;}

//////////////////////////////////////////////////////////////////////////////
//	if (fvarg3 != 0) {psPxu->b_lenv = 1;}

		// fixed kneed SH attack, for less noise.
		psPxu->b_lenv = b_inr / psPxu->b_lenv;
		double b_knee = b_inr; if (b_knee > 1) {b_knee = 1;}
		if (b_knee < psPxu->b_lenv) {psPxu->b_lenv = b_knee;}
		psPxu->b_lenv = b_inr / psPxu->b_lenv;

		psPxu->b_h2buf[psPxu->i_bfcn] = psPxu->b_lenv;
		if (psPxu->b_h2highest < psPxu->b_lenv) {
			psPxu->i_h2sortcurr = 0;
			psPxu->b_h2highest = psPxu->b_lenv;
		}
		else if (psPxu->i_h2sortcurr > i_htravlensub) {
			int cnt = 0;
			psPxu->b_h2highest = 0;
			while (cnt < i_htravlen) {
				int i_bufpoint = psPxu->i_bfcn - cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				if (psPxu->b_h2highest < psPxu->b_h2buf[i_bufpoint]) {
					psPxu->b_h2highest = psPxu->b_h2buf[i_bufpoint];
					psPxu->i_h2sortcurr = cnt;
				}
				cnt++;
			}
		}
		psPxu->i_h2sortcurr++;
		if (psPxu->i_h2sortcurr > 1002) {psPxu->i_h2sortcurr = 1002;}
		psPxu->b_lenv = psPxu->b_h2highest;

		psPxu->b_syncbufL[psPxu->i_bfcn] = b_in1;
		psPxu->b_syncbufR[psPxu->i_bfcn] = b_in2;
		b_in1 = psPxu->b_syncbufL[i_syncread];
		b_in2 = psPxu->b_syncbufR[i_syncread];

//////////////////////////////////
		// SH filter, pushed to the max.
		double b_lenvf = psPxu->b_lenv;
		double b_inrshf = b_inrsh;
		
//.. not implemented, mid-range sensitive envelope movement.
//		psPxu->b_lenvf = 1/psPxu->b_lenvf;
//		psPxu->b_lenvf = psPxu->b_lenvf + b_spoff;
//		psPxu->b_lenvf = psPxu->b_lenvf / b_spoff;
//		psPxu->b_lenvf = psPxu->b_lenvf - 1;
//		psPxu->b_lenvf = pow((psPxu->b_lenvf),b_spow1);
//
//		b_inrshf = 1/b_inrshf;		
//		b_inrshf = b_inrshf + b_spoff;
//		b_inrshf = b_inrshf / b_spoff;
//		b_inrshf = b_inrshf - 1;
//		b_inrshf = pow((b_inrshf),b_spow1);
//
//		b_f1 = exp(psPxu->b_lenvf);
//		b_f2 = b_f1 / psPxu->b_lenvf;
//		b_f3 = exp(b_inrshf);
//		b_f4 = b_f3 / b_inrshf;


		b_f1 = 1/b_lenvf;
		b_f1 = sqrt(b_f1);
		b_f2 = b_f1 / b_lenvf;
		
		b_f3 = 1/b_inrsh;
		b_f3 = sqrt(b_f3);
		b_f4 = b_f3 / b_inrshf;
		
		
		
		// Tiir ubkgauss59
		psPxu->b_avg1buf[psPxu->i_bfcn] = b_f1;
		psPxu->b_avg2buf[psPxu->i_bfcn] = b_f2;
		psPxu->b_avg3buf[psPxu->i_bfcn] = b_f3;
		psPxu->b_avg4buf[psPxu->i_bfcn] = b_f4;

		i_bufpoint = psPxu->i_bfcn - i_avglen;
		if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}

		b_sub = psPxu->b_avg1buf[i_bufpoint];

		psPxu->b_tg1 = psPxu->b_tg1 - (b_sub * psPxu->b_at1);
		psPxu->b_tg1 = psPxu->b_tg1 + ((-psPxu->b_tg1 + b_f1) * b_tgcut1);

		psPxu->b_tg2 = psPxu->b_tg2 - (b_sub * psPxu->b_at2);
		psPxu->b_tg2 = psPxu->b_tg2 + ((-psPxu->b_tg2 + b_f1) * b_tgcut2);

		psPxu->b_tg3 = psPxu->b_tg3 - (b_sub * psPxu->b_at3);
		psPxu->b_tg3 = psPxu->b_tg3 + ((-psPxu->b_tg3 + b_f1) * b_tgcut3);

		psPxu->b_tg4 = psPxu->b_tg4 - (b_sub * psPxu->b_at4);
		psPxu->b_tg4 = psPxu->b_tg4 + ((-psPxu->b_tg4 + b_f1) * b_tgcut4);

		psPxu->b_tg5 = psPxu->b_tg5 - (b_sub * psPxu->b_at5);
		psPxu->b_tg5 = psPxu->b_tg5 + ((-psPxu->b_tg5 + b_f1) * b_tgcut5);

		b_sub = psPxu->b_avg2buf[i_bufpoint];

		psPxu->b_tg6 = psPxu->b_tg6 - (b_sub * psPxu->b_at1);
		psPxu->b_tg6 = psPxu->b_tg6 + ((-psPxu->b_tg6 + b_f2) * b_tgcut1);

		psPxu->b_tg7 = psPxu->b_tg7 - (b_sub * psPxu->b_at2);
		psPxu->b_tg7 = psPxu->b_tg7 + ((-psPxu->b_tg7 + b_f2) * b_tgcut2);

		psPxu->b_tg8 = psPxu->b_tg8 - (b_sub * psPxu->b_at3);
		psPxu->b_tg8 = psPxu->b_tg8 + ((-psPxu->b_tg8 + b_f2) * b_tgcut3);

		psPxu->b_tg9 = psPxu->b_tg9 - (b_sub * psPxu->b_at4);
		psPxu->b_tg9 = psPxu->b_tg9 + ((-psPxu->b_tg9 + b_f2) * b_tgcut4);

		psPxu->b_tg10 = psPxu->b_tg10 - (b_sub * psPxu->b_at5);
		psPxu->b_tg10 = psPxu->b_tg10 + ((-psPxu->b_tg10 + b_f2) * b_tgcut5);

		b_sub = psPxu->b_avg3buf[i_bufpoint];

		psPxu->b_tg11 = psPxu->b_tg11 - (b_sub * psPxu->b_at1);
		psPxu->b_tg11 = psPxu->b_tg11 + ((-psPxu->b_tg11 + b_f3) * b_tgcut1);

		psPxu->b_tg12 = psPxu->b_tg12 - (b_sub * psPxu->b_at2);
		psPxu->b_tg12 = psPxu->b_tg12 + ((-psPxu->b_tg12 + b_f3) * b_tgcut2);

		psPxu->b_tg13 = psPxu->b_tg13 - (b_sub * psPxu->b_at3);
		psPxu->b_tg13 = psPxu->b_tg13 + ((-psPxu->b_tg13 + b_f3) * b_tgcut3);

		psPxu->b_tg14 = psPxu->b_tg14 - (b_sub * psPxu->b_at4);
		psPxu->b_tg14 = psPxu->b_tg14 + ((-psPxu->b_tg14 + b_f3) * b_tgcut4);

		psPxu->b_tg15 = psPxu->b_tg15 - (b_sub * psPxu->b_at5);
		psPxu->b_tg15 = psPxu->b_tg15 + ((-psPxu->b_tg15 + b_f3) * b_tgcut5);

		b_sub = psPxu->b_avg4buf[i_bufpoint];

		psPxu->b_tg16 = psPxu->b_tg16 - (b_sub * psPxu->b_at1);
		psPxu->b_tg16 = psPxu->b_tg16 + ((-psPxu->b_tg16 + b_f4) * b_tgcut1);

		psPxu->b_tg17 = psPxu->b_tg17 - (b_sub * psPxu->b_at2);
		psPxu->b_tg17 = psPxu->b_tg17 + ((-psPxu->b_tg17 + b_f4) * b_tgcut2);

		psPxu->b_tg18 = psPxu->b_tg18 - (b_sub * psPxu->b_at3);
		psPxu->b_tg18 = psPxu->b_tg18 + ((-psPxu->b_tg18 + b_f4) * b_tgcut3);

		psPxu->b_tg19 = psPxu->b_tg19 - (b_sub * psPxu->b_at4);
		psPxu->b_tg19 = psPxu->b_tg19 + ((-psPxu->b_tg19 + b_f4) * b_tgcut4);

		psPxu->b_tg20 = psPxu->b_tg20 - (b_sub * psPxu->b_at5);
		psPxu->b_tg20 = psPxu->b_tg20 + ((-psPxu->b_tg20 + b_f4) * b_tgcut5);


		double b_tgo1 = psPxu->b_tg1;
		double b_tgo2 = psPxu->b_tg2;
		double b_tgo3 = psPxu->b_tg3;
		double b_tgo4 = psPxu->b_tg4;
		double b_tgo5 = psPxu->b_tg5;
		double b_tgo6 = psPxu->b_tg6;
		double b_tgo7 = psPxu->b_tg7;
		double b_tgo8 = psPxu->b_tg8;
		double b_tgo9 = psPxu->b_tg9;
		double b_tgo10 = psPxu->b_tg10;
		double b_tgo11 = psPxu->b_tg11;
		double b_tgo12 = psPxu->b_tg12;
		double b_tgo13 = psPxu->b_tg13;
		double b_tgo14 = psPxu->b_tg14;
		double b_tgo15 = psPxu->b_tg15;
		double b_tgo16 = psPxu->b_tg16;
		double b_tgo17 = psPxu->b_tg17;
		double b_tgo18 = psPxu->b_tg18;
		double b_tgo19 = psPxu->b_tg19;
		double b_tgo20 = psPxu->b_tg20;

		psPxu->b_tg1 = psPxu->b_tg1 + (-psPxu->b_tg1 * b_tgcut1);
		psPxu->b_tg2 = psPxu->b_tg2 + (-psPxu->b_tg2 * b_tgcut2);
		psPxu->b_tg3 = psPxu->b_tg3 + (-psPxu->b_tg3 * b_tgcut3);
		psPxu->b_tg4 = psPxu->b_tg4 + (-psPxu->b_tg4 * b_tgcut4);
		psPxu->b_tg5 = psPxu->b_tg5 + (-psPxu->b_tg5 * b_tgcut5);
		psPxu->b_tg6 = psPxu->b_tg6 + (-psPxu->b_tg6 * b_tgcut1);
		psPxu->b_tg7 = psPxu->b_tg7 + (-psPxu->b_tg7 * b_tgcut2);
		psPxu->b_tg8 = psPxu->b_tg8 + (-psPxu->b_tg8 * b_tgcut3);
		psPxu->b_tg9 = psPxu->b_tg9 + (-psPxu->b_tg9 * b_tgcut4);
		psPxu->b_tg10 = psPxu->b_tg10 + (-psPxu->b_tg10 * b_tgcut5);
		psPxu->b_tg11 = psPxu->b_tg11 + (-psPxu->b_tg11 * b_tgcut1);
		psPxu->b_tg12 = psPxu->b_tg12 + (-psPxu->b_tg12 * b_tgcut2);
		psPxu->b_tg13 = psPxu->b_tg13 + (-psPxu->b_tg13 * b_tgcut3);
		psPxu->b_tg14 = psPxu->b_tg14 + (-psPxu->b_tg14 * b_tgcut4);
		psPxu->b_tg15 = psPxu->b_tg15 + (-psPxu->b_tg15 * b_tgcut5);
		psPxu->b_tg16 = psPxu->b_tg16 + (-psPxu->b_tg16 * b_tgcut1);
		psPxu->b_tg17 = psPxu->b_tg17 + (-psPxu->b_tg17 * b_tgcut2);
		psPxu->b_tg18 = psPxu->b_tg18 + (-psPxu->b_tg18 * b_tgcut3);
		psPxu->b_tg19 = psPxu->b_tg19 + (-psPxu->b_tg19 * b_tgcut4);
		psPxu->b_tg20 = psPxu->b_tg20 + (-psPxu->b_tg20 * b_tgcut5);

		b_tgo1 = b_tgo1+ psPxu->b_tg1;
		b_tgo2 = b_tgo2+ psPxu->b_tg2;
		b_tgo3 = b_tgo3+ psPxu->b_tg3;
		b_tgo4 = b_tgo4+ psPxu->b_tg4;
		b_tgo5 = b_tgo5+ psPxu->b_tg5;
		b_tgo6 = b_tgo6+ psPxu->b_tg6;
		b_tgo7 = b_tgo7+ psPxu->b_tg7;
		b_tgo8 = b_tgo8+ psPxu->b_tg8;
		b_tgo9 = b_tgo9+ psPxu->b_tg9;
		b_tgo10 =b_tgo10+ psPxu->b_tg10;
		b_tgo11 =b_tgo11+ psPxu->b_tg11;
		b_tgo12 =b_tgo12+ psPxu->b_tg12;
		b_tgo13 =b_tgo13+ psPxu->b_tg13;
		b_tgo14 =b_tgo14+ psPxu->b_tg14;
		b_tgo15 =b_tgo15+ psPxu->b_tg15;
		b_tgo16 =b_tgo16+ psPxu->b_tg16;
		b_tgo17 =b_tgo17+ psPxu->b_tg17;
		b_tgo18 =b_tgo18+ psPxu->b_tg18;
		b_tgo19 =b_tgo19+ psPxu->b_tg19;
		b_tgo20 =b_tgo20+ psPxu->b_tg20;

		psPxu->b_tg1 = psPxu->b_tg1 + (-psPxu->b_tg1 * b_tgcut1);
		psPxu->b_tg2 = psPxu->b_tg2 + (-psPxu->b_tg2 * b_tgcut2);
		psPxu->b_tg3 = psPxu->b_tg3 + (-psPxu->b_tg3 * b_tgcut3);
		psPxu->b_tg4 = psPxu->b_tg4 + (-psPxu->b_tg4 * b_tgcut4);
		psPxu->b_tg5 = psPxu->b_tg5 + (-psPxu->b_tg5 * b_tgcut5);
		psPxu->b_tg6 = psPxu->b_tg6 + (-psPxu->b_tg6 * b_tgcut1);
		psPxu->b_tg7 = psPxu->b_tg7 + (-psPxu->b_tg7 * b_tgcut2);
		psPxu->b_tg8 = psPxu->b_tg8 + (-psPxu->b_tg8 * b_tgcut3);
		psPxu->b_tg9 = psPxu->b_tg9 + (-psPxu->b_tg9 * b_tgcut4);
		psPxu->b_tg10 = psPxu->b_tg10 + (-psPxu->b_tg10 * b_tgcut5);
		psPxu->b_tg11 = psPxu->b_tg11 + (-psPxu->b_tg11 * b_tgcut1);
		psPxu->b_tg12 = psPxu->b_tg12 + (-psPxu->b_tg12 * b_tgcut2);
		psPxu->b_tg13 = psPxu->b_tg13 + (-psPxu->b_tg13 * b_tgcut3);
		psPxu->b_tg14 = psPxu->b_tg14 + (-psPxu->b_tg14 * b_tgcut4);
		psPxu->b_tg15 = psPxu->b_tg15 + (-psPxu->b_tg15 * b_tgcut5);
		psPxu->b_tg16 = psPxu->b_tg16 + (-psPxu->b_tg16 * b_tgcut1);
		psPxu->b_tg17 = psPxu->b_tg17 + (-psPxu->b_tg17 * b_tgcut2);
		psPxu->b_tg18 = psPxu->b_tg18 + (-psPxu->b_tg18 * b_tgcut3);
		psPxu->b_tg19 = psPxu->b_tg19 + (-psPxu->b_tg19 * b_tgcut4);
		psPxu->b_tg20 = psPxu->b_tg20 + (-psPxu->b_tg20 * b_tgcut5);

		b_tgo1 = b_tgo1+ psPxu->b_tg1;
		b_tgo2 = b_tgo2+ psPxu->b_tg2;
		b_tgo3 = b_tgo3+ psPxu->b_tg3;
		b_tgo4 = b_tgo4+ psPxu->b_tg4;
		b_tgo5 = b_tgo5+ psPxu->b_tg5;
		b_tgo6 = b_tgo6+ psPxu->b_tg6;
		b_tgo7 = b_tgo7+ psPxu->b_tg7;
		b_tgo8 = b_tgo8+ psPxu->b_tg8;
		b_tgo9 = b_tgo9+ psPxu->b_tg9;
		b_tgo10 =b_tgo10+ psPxu->b_tg10;
		b_tgo11 =b_tgo11+ psPxu->b_tg11;
		b_tgo12 =b_tgo12+ psPxu->b_tg12;
		b_tgo13 =b_tgo13+ psPxu->b_tg13;
		b_tgo14 =b_tgo14+ psPxu->b_tg14;
		b_tgo15 =b_tgo15+ psPxu->b_tg15;
		b_tgo16 =b_tgo16+ psPxu->b_tg16;
		b_tgo17 =b_tgo17+ psPxu->b_tg17;
		b_tgo18 =b_tgo18+ psPxu->b_tg18;
		b_tgo19 =b_tgo19+ psPxu->b_tg19;
		b_tgo20 =b_tgo20+ psPxu->b_tg20;


		
//psPxu->i_optcnt = i_optimize; // run unoptimized.
		if (psPxu->i_optcnt == i_optimize) {
			double b_one;
			// find at
			psPxu->i_cnt = 0;
			psPxu->b_nr1 = 0;
			psPxu->b_at1 = 0;
			while (psPxu->i_cnt < i_avglen) {
				if (psPxu->i_cnt == 0) {psPxu->b_at1 = psPxu->b_at1 + ((-psPxu->b_at1 + 1) * b_tgcut1);psPxu->b_at1 = psPxu->b_at1 + ((-psPxu->b_at1 + 0) * b_tgcut1);psPxu->b_at1 = psPxu->b_at1 + ((-psPxu->b_at1 + 0) * b_tgcut1);} else {psPxu->b_at1 = psPxu->b_at1 + ((-psPxu->b_at1 + 0) * b_tgcut1);psPxu->b_at1 = psPxu->b_at1 + ((-psPxu->b_at1 + 0) * b_tgcut1);psPxu->b_at1 = psPxu->b_at1 + ((-psPxu->b_at1 + 0) * b_tgcut1);}
				psPxu->b_nr1 = psPxu->b_nr1 + ((-psPxu->b_nr1 + 1) * b_tgcut1);
				psPxu->b_nr1 = psPxu->b_nr1 + ((-psPxu->b_nr1 + 0) * b_tgcut1);
				psPxu->b_nr1 = psPxu->b_nr1 + ((-psPxu->b_nr1 + 0) * b_tgcut1);
				psPxu->i_cnt++;
			}

		// make filter correct with regards to previous values
			psPxu->i_cnt = i_avglen-1;
			psPxu->b_tg1 = 0;
			psPxu->b_tg6 = 0;
			psPxu->b_tg11 = 0;
			psPxu->b_tg16 = 0;
			while (psPxu->i_cnt > -1) {
				i_bufpoint = psPxu->i_bfcn - psPxu->i_cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				psPxu->b_tg1 = psPxu->b_tg1 + ((-psPxu->b_tg1 + psPxu->b_avg1buf[i_bufpoint]) * b_tgcut1);
				psPxu->b_tg6 = psPxu->b_tg6 + ((-psPxu->b_tg6 + psPxu->b_avg2buf[i_bufpoint]) * b_tgcut1);
				psPxu->b_tg11= psPxu->b_tg11+ ((-psPxu->b_tg11+ psPxu->b_avg3buf[i_bufpoint]) * b_tgcut1);
				psPxu->b_tg16= psPxu->b_tg16+ ((-psPxu->b_tg16+ psPxu->b_avg4buf[i_bufpoint]) * b_tgcut1);
				psPxu->b_tg1 = psPxu->b_tg1 + ((-psPxu->b_tg1 ) * b_tgcut1);
				psPxu->b_tg6 = psPxu->b_tg6 + ((-psPxu->b_tg6 ) * b_tgcut1);
				psPxu->b_tg11= psPxu->b_tg11+ ((-psPxu->b_tg11) * b_tgcut1);
				psPxu->b_tg16= psPxu->b_tg16+ ((-psPxu->b_tg16) * b_tgcut1);
				psPxu->b_tg1 = psPxu->b_tg1 + ((-psPxu->b_tg1 ) * b_tgcut1);
				psPxu->b_tg6 = psPxu->b_tg6 + ((-psPxu->b_tg6 ) * b_tgcut1);
				psPxu->b_tg11= psPxu->b_tg11+ ((-psPxu->b_tg11) * b_tgcut1);
				psPxu->b_tg16= psPxu->b_tg16+ ((-psPxu->b_tg16) * b_tgcut1);
				psPxu->i_cnt--;
			}

		// find at
			psPxu->i_cnt = 0;
			psPxu->b_nr2 = 0;
			psPxu->b_at2 = 0;
			while (psPxu->i_cnt < i_avglen) {
				if (psPxu->i_cnt == 0) {psPxu->b_at2 = psPxu->b_at2 + ((-psPxu->b_at2 + 1) * b_tgcut2);psPxu->b_at2 = psPxu->b_at2 + ((-psPxu->b_at2 + 0) * b_tgcut2);psPxu->b_at2 = psPxu->b_at2 + ((-psPxu->b_at2 + 0) * b_tgcut2);} else {psPxu->b_at2 = psPxu->b_at2 + ((-psPxu->b_at2 + 0) * b_tgcut2);psPxu->b_at2 = psPxu->b_at2 + ((-psPxu->b_at2 + 0) * b_tgcut2);psPxu->b_at2 = psPxu->b_at2 + ((-psPxu->b_at2 + 0) * b_tgcut2);}
				psPxu->b_nr2 = psPxu->b_nr2 + ((-psPxu->b_nr2 + 1) * b_tgcut2);
				psPxu->b_nr2 = psPxu->b_nr2 + ((-psPxu->b_nr2 + 0) * b_tgcut2);
				psPxu->b_nr2 = psPxu->b_nr2 + ((-psPxu->b_nr2 + 0) * b_tgcut2);
				psPxu->i_cnt++;
			}

		// make filter correct with regards to previous values
			psPxu->i_cnt = i_avglen-1;
			psPxu->b_tg2 = 0;
			psPxu->b_tg7 = 0;
			psPxu->b_tg12 = 0;
			psPxu->b_tg17 = 0;
			while (psPxu->i_cnt > -1) {
				i_bufpoint = psPxu->i_bfcn - psPxu->i_cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				psPxu->b_tg2 = psPxu->b_tg2 + ((-psPxu->b_tg2 + psPxu->b_avg1buf[i_bufpoint]) * b_tgcut2);
				psPxu->b_tg7 = psPxu->b_tg7 + ((-psPxu->b_tg7 + psPxu->b_avg2buf[i_bufpoint]) * b_tgcut2);
				psPxu->b_tg12= psPxu->b_tg12+ ((-psPxu->b_tg12+ psPxu->b_avg3buf[i_bufpoint]) * b_tgcut2);
				psPxu->b_tg17= psPxu->b_tg17+ ((-psPxu->b_tg17+ psPxu->b_avg4buf[i_bufpoint]) * b_tgcut2);
				psPxu->b_tg2 = psPxu->b_tg2 + ((-psPxu->b_tg2 ) * b_tgcut2);
				psPxu->b_tg7 = psPxu->b_tg7 + ((-psPxu->b_tg7 ) * b_tgcut2);
				psPxu->b_tg12= psPxu->b_tg12+ ((-psPxu->b_tg12) * b_tgcut2);
				psPxu->b_tg17= psPxu->b_tg17+ ((-psPxu->b_tg17) * b_tgcut2);
				psPxu->b_tg2 = psPxu->b_tg2 + ((-psPxu->b_tg2 ) * b_tgcut2);
				psPxu->b_tg7 = psPxu->b_tg7 + ((-psPxu->b_tg7 ) * b_tgcut2);
				psPxu->b_tg12= psPxu->b_tg12+ ((-psPxu->b_tg12) * b_tgcut2);
				psPxu->b_tg17= psPxu->b_tg17+ ((-psPxu->b_tg17) * b_tgcut2);
				psPxu->i_cnt--;
			}

		// find at
			psPxu->i_cnt = 0;
			psPxu->b_nr3 = 0;
			psPxu->b_at3 = 0;
			while (psPxu->i_cnt < i_avglen) {
				if (psPxu->i_cnt == 0) {psPxu->b_at3 = psPxu->b_at3 + ((-psPxu->b_at3 + 1) * b_tgcut3);psPxu->b_at3 = psPxu->b_at3 + ((-psPxu->b_at3 + 0) * b_tgcut3);psPxu->b_at3 = psPxu->b_at3 + ((-psPxu->b_at3 + 0) * b_tgcut3);} else {psPxu->b_at3 = psPxu->b_at3 + ((-psPxu->b_at3 + 0) * b_tgcut3);psPxu->b_at3 = psPxu->b_at3 + ((-psPxu->b_at3 + 0) * b_tgcut3);psPxu->b_at3 = psPxu->b_at3 + ((-psPxu->b_at3 + 0) * b_tgcut3);}
				psPxu->b_nr3 = psPxu->b_nr3 + ((-psPxu->b_nr3 + 1) * b_tgcut3);
				psPxu->b_nr3 = psPxu->b_nr3 + ((-psPxu->b_nr3 + 0) * b_tgcut3);
				psPxu->b_nr3 = psPxu->b_nr3 + ((-psPxu->b_nr3 + 0) * b_tgcut3);
				psPxu->i_cnt++;
			}

		// make filter correct with regards to previous values
			psPxu->i_cnt = i_avglen-1;
			psPxu->b_tg3 = 0;
			psPxu->b_tg8 = 0;
			psPxu->b_tg13 = 0;
			psPxu->b_tg18 = 0;
			while (psPxu->i_cnt > -1) {
				i_bufpoint = psPxu->i_bfcn - psPxu->i_cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				psPxu->b_tg3 = psPxu->b_tg3 + ((-psPxu->b_tg3 + psPxu->b_avg1buf[i_bufpoint]) * b_tgcut3);
				psPxu->b_tg8 = psPxu->b_tg8 + ((-psPxu->b_tg8 + psPxu->b_avg2buf[i_bufpoint]) * b_tgcut3);
				psPxu->b_tg13= psPxu->b_tg13+ ((-psPxu->b_tg13+ psPxu->b_avg3buf[i_bufpoint]) * b_tgcut3);
				psPxu->b_tg18= psPxu->b_tg18+ ((-psPxu->b_tg18+ psPxu->b_avg4buf[i_bufpoint]) * b_tgcut3);
				psPxu->b_tg3 = psPxu->b_tg3 + ((-psPxu->b_tg3 ) * b_tgcut3);
				psPxu->b_tg8 = psPxu->b_tg8 + ((-psPxu->b_tg8 ) * b_tgcut3);
				psPxu->b_tg13= psPxu->b_tg13+ ((-psPxu->b_tg13) * b_tgcut3);
				psPxu->b_tg18= psPxu->b_tg18+ ((-psPxu->b_tg18) * b_tgcut3);
				psPxu->b_tg3 = psPxu->b_tg3 + ((-psPxu->b_tg3 ) * b_tgcut3);
				psPxu->b_tg8 = psPxu->b_tg8 + ((-psPxu->b_tg8 ) * b_tgcut3);
				psPxu->b_tg13= psPxu->b_tg13+ ((-psPxu->b_tg13) * b_tgcut3);
				psPxu->b_tg18= psPxu->b_tg18+ ((-psPxu->b_tg18) * b_tgcut3);
				psPxu->i_cnt--;
			}

		// find at
			psPxu->i_cnt = 0;
			psPxu->b_nr4 = 0;
			psPxu->b_at4 = 0;
			while (psPxu->i_cnt < i_avglen) {
				if (psPxu->i_cnt == 0) {psPxu->b_at4 = psPxu->b_at4 + ((-psPxu->b_at4 + 1) * b_tgcut4);psPxu->b_at4 = psPxu->b_at4 + ((-psPxu->b_at4 + 0) * b_tgcut4);psPxu->b_at4 = psPxu->b_at4 + ((-psPxu->b_at4 + 0) * b_tgcut4);} else {psPxu->b_at4 = psPxu->b_at4 + ((-psPxu->b_at4 + 0) * b_tgcut4);psPxu->b_at4 = psPxu->b_at4 + ((-psPxu->b_at4 + 0) * b_tgcut4);psPxu->b_at4 = psPxu->b_at4 + ((-psPxu->b_at4 + 0) * b_tgcut4);}
				psPxu->b_nr4 = psPxu->b_nr4 + ((-psPxu->b_nr4 + 1) * b_tgcut4);
				psPxu->b_nr4 = psPxu->b_nr4 + ((-psPxu->b_nr4 + 0) * b_tgcut4);
				psPxu->b_nr4 = psPxu->b_nr4 + ((-psPxu->b_nr4 + 0) * b_tgcut4);
				psPxu->i_cnt++;
			}

		// make filter correct with regards to previous values
			psPxu->i_cnt = i_avglen-1;
			psPxu->b_tg4 = 0;
			psPxu->b_tg9 = 0;
			psPxu->b_tg14 = 0;
			psPxu->b_tg19 = 0;
			while (psPxu->i_cnt > -1) {
				i_bufpoint = psPxu->i_bfcn - psPxu->i_cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				psPxu->b_tg4 = psPxu->b_tg4 + ((-psPxu->b_tg4 + psPxu->b_avg1buf[i_bufpoint]) * b_tgcut4);
				psPxu->b_tg9 = psPxu->b_tg9 + ((-psPxu->b_tg9 + psPxu->b_avg2buf[i_bufpoint]) * b_tgcut4);
				psPxu->b_tg14= psPxu->b_tg14+ ((-psPxu->b_tg14+ psPxu->b_avg3buf[i_bufpoint]) * b_tgcut4);
				psPxu->b_tg19= psPxu->b_tg19+ ((-psPxu->b_tg19+ psPxu->b_avg4buf[i_bufpoint]) * b_tgcut4);
				psPxu->b_tg4 = psPxu->b_tg4 + ((-psPxu->b_tg4 ) * b_tgcut4);
				psPxu->b_tg9 = psPxu->b_tg9 + ((-psPxu->b_tg9 ) * b_tgcut4);
				psPxu->b_tg14= psPxu->b_tg14+ ((-psPxu->b_tg14) * b_tgcut4);
				psPxu->b_tg19= psPxu->b_tg19+ ((-psPxu->b_tg19) * b_tgcut4);
				psPxu->b_tg4 = psPxu->b_tg4 + ((-psPxu->b_tg4 ) * b_tgcut4);
				psPxu->b_tg9 = psPxu->b_tg9 + ((-psPxu->b_tg9 ) * b_tgcut4);
				psPxu->b_tg14= psPxu->b_tg14+ ((-psPxu->b_tg14) * b_tgcut4);
				psPxu->b_tg19= psPxu->b_tg19+ ((-psPxu->b_tg19) * b_tgcut4);
				psPxu->i_cnt--;
			}

		// find at
			psPxu->i_cnt = 0;
			psPxu->b_nr5 = 0;
			psPxu->b_at5 = 0;
			while (psPxu->i_cnt < i_avglen) {
				if (psPxu->i_cnt == 0) {psPxu->b_at5 = psPxu->b_at5 + ((-psPxu->b_at5 + 1) * b_tgcut5);psPxu->b_at5 = psPxu->b_at5 + ((-psPxu->b_at5 + 0) * b_tgcut5);psPxu->b_at5 = psPxu->b_at5 + ((-psPxu->b_at5 + 0) * b_tgcut5);} else {psPxu->b_at5 = psPxu->b_at5 + ((-psPxu->b_at5 + 0) * b_tgcut5);psPxu->b_at5 = psPxu->b_at5 + ((-psPxu->b_at5 + 0) * b_tgcut5);psPxu->b_at5 = psPxu->b_at5 + ((-psPxu->b_at5 + 0) * b_tgcut5);}
				psPxu->b_nr5 = psPxu->b_nr5 + ((-psPxu->b_nr5 + 1) * b_tgcut5);
				psPxu->b_nr5 = psPxu->b_nr5 + ((-psPxu->b_nr5 + 0) * b_tgcut5);
				psPxu->b_nr5 = psPxu->b_nr5 + ((-psPxu->b_nr5 + 0) * b_tgcut5);
				psPxu->i_cnt++;
			}

		// make filter correct with regards to previous values
			psPxu->i_cnt = i_avglen-1;
			psPxu->b_tg5 = 0;
			psPxu->b_tg10 = 0;
			psPxu->b_tg15 = 0;
			psPxu->b_tg20 = 0;
			b_one = b_tgcut5;
			while (psPxu->i_cnt > -1) {
				i_bufpoint = psPxu->i_bfcn - psPxu->i_cnt;
				if (i_bufpoint < 0) {i_bufpoint = 1002 + i_bufpoint;}
				psPxu->b_tg5 = psPxu->b_tg5 + ((-psPxu->b_tg5 + psPxu->b_avg1buf[i_bufpoint]) * b_tgcut5);
				psPxu->b_tg10= psPxu->b_tg10+ ((-psPxu->b_tg10+ psPxu->b_avg2buf[i_bufpoint]) * b_tgcut5);
				psPxu->b_tg15= psPxu->b_tg15+ ((-psPxu->b_tg15+ psPxu->b_avg3buf[i_bufpoint]) * b_tgcut5);
				psPxu->b_tg20= psPxu->b_tg20+ ((-psPxu->b_tg20+ psPxu->b_avg4buf[i_bufpoint]) * b_tgcut5);
				psPxu->b_tg5 = psPxu->b_tg5 + ((-psPxu->b_tg5 ) * b_tgcut5);
				psPxu->b_tg10= psPxu->b_tg10+ ((-psPxu->b_tg10) * b_tgcut5);
				psPxu->b_tg15= psPxu->b_tg15+ ((-psPxu->b_tg15) * b_tgcut5);
				psPxu->b_tg20= psPxu->b_tg20+ ((-psPxu->b_tg20) * b_tgcut5);
				psPxu->b_tg5 = psPxu->b_tg5 + ((-psPxu->b_tg5 ) * b_tgcut5);
				psPxu->b_tg10= psPxu->b_tg10+ ((-psPxu->b_tg10) * b_tgcut5);
				psPxu->b_tg15= psPxu->b_tg15+ ((-psPxu->b_tg15) * b_tgcut5);
				psPxu->b_tg20= psPxu->b_tg20+ ((-psPxu->b_tg20) * b_tgcut5);
				psPxu->i_cnt--;
			}
		}
		

// Compensated summing, for the extreme offsetted data.

				
		b_of1 = b_tggain1 * b_tgo1;
		b_of2 = b_tggain2 * b_tgo2;
		b_of3 = b_tggain3 * b_tgo3;
		b_of4 = b_tggain4 * b_tgo4;
		b_of5 = b_tggain5 * b_tgo5;
		
		b_cErr = 0;
		b_cSum = 0;

		b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_f1 = b_cSum;
///////////////////////////////////////////

		b_of1 = b_tggain1 * b_tgo6;
		b_of2 = b_tggain2 * b_tgo7;
		b_of3 = b_tggain3 * b_tgo8;
		b_of4 = b_tggain4 * b_tgo9;
		b_of5 = b_tggain5 * b_tgo10;

		b_cErr = 0;
		b_cSum = 0;

		b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_f2 = b_cSum;
///////////////////////////////////////////

		b_of1 = b_tggain1 * b_tgo11;
		b_of2 = b_tggain2 * b_tgo12;
		b_of3 = b_tggain3 * b_tgo13;
		b_of4 = b_tggain4 * b_tgo14;
		b_of5 = b_tggain5 * b_tgo15;

		b_cErr = 0;
		b_cSum = 0;
	
		b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_f3 = b_cSum;
///////////////////////////////////////////

		b_of1 = b_tggain1 * b_tgo16;
		b_of2 = b_tggain2 * b_tgo17;
		b_of3 = b_tggain3 * b_tgo18;
		b_of4 = b_tggain4 * b_tgo19;
		b_of5 = b_tggain5 * b_tgo20;

		b_cErr = 0;
		b_cSum = 0;

		b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_f4 = b_cSum;
///////////////////////////////////////////

		b_of1 = b_tggain1 * psPxu->b_nr1;
		b_of2 = b_tggain2 * psPxu->b_nr2;
		b_of3 = b_tggain3 * psPxu->b_nr3;
		b_of4 = b_tggain4 * psPxu->b_nr4;
		b_of5 = b_tggain5 * psPxu->b_nr5;

		b_cErr = 0;
		b_cSum = 0;

		b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		double b_fn = b_cSum;
	
		b_f1 = b_f1 / b_fn;
		b_f2 = b_f2 / b_fn;
		b_f3 = b_f3 / b_fn;
		b_f4 = b_f4 / b_fn;
		
///////////////////////////////////////////


// small window for discontinuities
		int i_discread = psPxu->i_bfcn - i_la; 
		if (i_discread < 0) {i_discread = 1002 + i_discread;}

		double b_disc1 = psPxu->b_avg1buf[i_syncread];
		double b_disc2 = psPxu->b_avg2buf[i_syncread];
		double b_disc3 = psPxu->b_avg3buf[i_syncread];
		double b_disc4 = psPxu->b_avg4buf[i_syncread];
		

		b_f1 = b_f1 + ((-b_f1 + b_disc1) * b_discam);
		b_f2 = b_f2 + ((-b_f2 + b_disc2) * b_discam);
		b_f3 = b_f3 + ((-b_f3 + b_disc3) * b_discam);
		b_f4 = b_f4 + ((-b_f4 + b_disc4) * b_discam);

////////////////////////////////////////////////////////////////////////////////////////

		b_lenvf = (b_f1 / b_f2);
		b_inrshf = (b_f3 / b_f4);
		
//.. not implemented, mid-range sensitive envelope movement.
//
//		b_prefer = b_prefer + b_spoff2;
//		b_prefer = b_prefer / b_spoff2;
//		b_prefer = b_prefer - 1;
//		b_prefer = pow(b_prefer,b_spow3);
//
//		psPxu->b_lenvf = pow(b_prefer,b_spow1) / psPxu->b_lenvf;
//		psPxu->b_lenvf = pow(psPxu->b_lenvf,b_spow2);
//		psPxu->b_lenvf = b_prefer / psPxu->b_lenvf;
//		psPxu->b_lenvf = psPxu->b_lenvf + 1;
//		psPxu->b_lenvf = psPxu->b_lenvf * b_spoff;
//		psPxu->b_lenvf = psPxu->b_lenvf - b_spoff;
//		psPxu->b_lenvf = 1/psPxu->b_lenvf;
//
//		b_inrshf = pow(b_prefer,b_spow1) / b_inrshf;
//		b_inrshf = pow(b_inrshf,b_spow2);
//		b_inrshf = b_prefer / b_inrshf;
//		b_inrshf = b_inrshf + 1;
//		b_inrshf = b_inrshf * b_spoff;
//		b_inrshf = b_inrshf - b_spoff;
//		b_inrshf = 1/b_inrshf;


		// Do limiting.
		b_in1 = b_in1 / b_lenvf;
		b_in2 = b_in2 / b_lenvf;

////////////////////////////////////////////////////////////////////////////////////////
//////////////Release Stage

		// Gaussian Env Smoother / Denoiser / Saturated Attack "ubkgauss59"
		psPxu->b_lenv = psPxu->b_lenv*psPxu->b_lenv; // rms

		psPxu->b_gse1buf = psPxu->b_gse1buf + ((-psPxu->b_gse1buf + psPxu->b_lenv) * b_gse1cut); // onepoles.
		psPxu->b_gse2buf = psPxu->b_gse2buf + ((-psPxu->b_gse2buf + psPxu->b_lenv) * b_gse2cut);
		psPxu->b_gse3buf = psPxu->b_gse3buf + ((-psPxu->b_gse3buf + psPxu->b_lenv) * b_gse3cut);
		psPxu->b_gse4buf = psPxu->b_gse4buf + ((-psPxu->b_gse4buf + psPxu->b_lenv) * b_gse4cut);
		psPxu->b_gse5buf = psPxu->b_gse5buf + ((-psPxu->b_gse5buf + psPxu->b_lenv) * b_gse5cut);

		b_of1 = b_gse1gain * psPxu->b_gse1buf;
		b_of2 = b_gse2gain * psPxu->b_gse2buf;
		b_of3 = b_gse3gain * psPxu->b_gse3buf;
		b_of4 = b_gse4gain * psPxu->b_gse4buf;
		b_of5 = b_gse5gain * psPxu->b_gse5buf;

		b_cErr = 0;
		b_cSum = 0;

		b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

		b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		psPxu->b_lenv = b_cSum;

		psPxu->b_lenv = sqrt(psPxu->b_lenv);

		// release
//		b_rms = b_inrshf + ((-b_inrshf + b_rms) * b_rtype); // mix a little for psychoacoustic improvement. - not implemented.
		double b_deenv = b_rms;

		double b_lenvdet = b_inrsync / psPxu->b_lenv;
		double b_deenvdet = b_inrsync / b_deenv;
		b_lenvdet = b_lenvdet + (b_deenvdet * b_release); // release, in-phase.

		psPxu->b_lenv = b_inrsync / b_lenvdet;

		// releaselimit
		if (psPxu->b_lenv < b_inrshf) {psPxu->b_lenv = b_inrshf;} // not over filtered version.
		  
////////////////////////////////////////////////////////////////////////////////////////////////

//		UnNormalize
//		if (i_norm == 0) {
////////////////////////////////////////////////////////////////////////////////////////////////
		if (psPxu->i_optcnt == i_optimize) {psPxu->i_optcnt = 0;}

		psPxu->i_bfcn++;
		if (psPxu->i_bfcn == 1002) {psPxu->i_bfcn = 0;}


   *(pfOutput1++) = b_in1;
   *(pfOutput2++) = b_in2;

   *(pfLatency++) = i_la; // PDC LADSPA style.
 }


}

/*****************************************************************************/

/* Throw away a simple delay line. */
void 
cleanupPxu(LADSPA_Handle Instance) {
  free(Instance);
}

/*****************************************************************************/

LADSPA_Descriptor * g_psMonoDescriptor = NULL;
LADSPA_Descriptor * g_psStereoDescriptor = NULL;

/*****************************************************************************/

/* _init() is called automatically when the plugin library is first
   loaded. */
void 
_init() {

  char ** pcPortNames;
  LADSPA_PortDescriptor * piPortDescriptors;
  LADSPA_PortRangeHint * psPortRangeHints;

  g_psMonoDescriptor
    = (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));
  g_psStereoDescriptor 
    = (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

  if (g_psMonoDescriptor) {
  
    g_psMonoDescriptor->UniqueID
      = 998;                                         	
    g_psMonoDescriptor->Label
      = strdup("Mlm_Limiter_M");				
    g_psMonoDescriptor->Properties
      = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    g_psMonoDescriptor->Name 
      = strdup("Millennium Limiter [mono] (not implemented)");			
    g_psMonoDescriptor->Maker
      = strdup("Opensource GPL");
    g_psMonoDescriptor->Copyright
      = strdup("Opensource GPL");
    g_psMonoDescriptor->PortCount
      = 5;						
    piPortDescriptors
      = (LADSPA_PortDescriptor *)calloc(5, sizeof(LADSPA_PortDescriptor));
    g_psMonoDescriptor->PortDescriptors
      = (const LADSPA_PortDescriptor *)piPortDescriptors;
    piPortDescriptors[PXU_CONTROL1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL3]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_INPUT1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT1]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    pcPortNames
      = (char **)calloc(5, sizeof(char *));
    g_psMonoDescriptor->PortNames 
      = (const char **)pcPortNames;
    pcPortNames[PXU_CONTROL1]
      = strdup("Gain");
    pcPortNames[PXU_CONTROL2]
      = strdup("Buffer");
    pcPortNames[PXU_CONTROL3]
      = strdup("Release");
    pcPortNames[PXU_INPUT1]
      = strdup("Input");
    pcPortNames[PXU_OUTPUT1]
      = strdup("Output");
    psPortRangeHints = ((LADSPA_PortRangeHint *)
			calloc(5, sizeof(LADSPA_PortRangeHint)));
    g_psMonoDescriptor->PortRangeHints
      = (const LADSPA_PortRangeHint *)psPortRangeHints;
    psPortRangeHints[PXU_CONTROL1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_CONTROL2].HintDescriptor
      = 0;
    psPortRangeHints[PXU_CONTROL3].HintDescriptor
      = 0;
    psPortRangeHints[PXU_INPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT1].HintDescriptor
      = 0;
    g_psMonoDescriptor->instantiate 
      = instantiatePxu;
    g_psMonoDescriptor->connect_port 
      = connectPortToPxu;
    g_psMonoDescriptor->activate
      = NULL;
    g_psMonoDescriptor->run
      = runMonoPxu;
    g_psMonoDescriptor->run_adding
      = NULL;
    g_psMonoDescriptor->set_run_adding_gain
      = NULL;
    g_psMonoDescriptor->deactivate
      = NULL;
    g_psMonoDescriptor->cleanup
      = cleanupPxu;
  }
  
  if (g_psStereoDescriptor) {
    
    g_psStereoDescriptor->UniqueID
      = 999;
    g_psStereoDescriptor->Label
      = strdup("Mlm_Limiter_S");
    g_psStereoDescriptor->Properties
      = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    g_psStereoDescriptor->Name 
      = strdup("Millennium Limiter [stereo]");
    g_psStereoDescriptor->Maker
      = strdup("Opensource GPL)");
    g_psStereoDescriptor->Copyright
      = strdup("Opensource GPL");
    g_psStereoDescriptor->PortCount
      = 8;
    piPortDescriptors
      = (LADSPA_PortDescriptor *)calloc(8, sizeof(LADSPA_PortDescriptor));
    g_psStereoDescriptor->PortDescriptors
      = (const LADSPA_PortDescriptor *)piPortDescriptors;
    piPortDescriptors[PXU_CONTROL1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL3]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_INPUT1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT1]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_INPUT2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT2]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_LATENCY]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    pcPortNames
      = (char **)calloc(8, sizeof(char *));
    g_psStereoDescriptor->PortNames 
      = (const char **)pcPortNames;
    pcPortNames[PXU_CONTROL1]
      = strdup("Gain");
    pcPortNames[PXU_CONTROL2]
      = strdup("Buffer");
    pcPortNames[PXU_CONTROL3]
      = strdup("Release");
    pcPortNames[PXU_INPUT1]
      = strdup("Input (Left)");
    pcPortNames[PXU_OUTPUT1]
      = strdup("Output (Left)");
    pcPortNames[PXU_INPUT2]
      = strdup("Input (Right)");
    pcPortNames[PXU_OUTPUT2]
      = strdup("Output (Right)");
    pcPortNames[PXU_LATENCY]
      = strdup("latency");
    psPortRangeHints = ((LADSPA_PortRangeHint *)
			calloc(8, sizeof(LADSPA_PortRangeHint)));
    g_psStereoDescriptor->PortRangeHints
      = (const LADSPA_PortRangeHint *)psPortRangeHints;

    psPortRangeHints[PXU_CONTROL1].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL1].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL1].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL2].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL2].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL2].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL3].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL3].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL3].UpperBound
      = 1;

    psPortRangeHints[PXU_INPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_INPUT2].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT2].HintDescriptor
      = 0;
    g_psStereoDescriptor->instantiate 
      = instantiatePxu;
    g_psStereoDescriptor->connect_port 
      = connectPortToPxu;
    g_psStereoDescriptor->activate
      = NULL;
    g_psStereoDescriptor->run
      = runStereoPxu;
    g_psStereoDescriptor->run_adding
      = NULL;
    g_psStereoDescriptor->set_run_adding_gain
      = NULL;
    g_psStereoDescriptor->deactivate
      = NULL;
    g_psStereoDescriptor->cleanup
      = cleanupPxu;
  }
}

/*****************************************************************************/

void
deleteDescriptor(LADSPA_Descriptor * psDescriptor) {
  unsigned long lIndex;
  if (psDescriptor) {
    free((char *)psDescriptor->Label);
    free((char *)psDescriptor->Name);
    free((char *)psDescriptor->Maker);
    free((char *)psDescriptor->Copyright);
    free((LADSPA_PortDescriptor *)psDescriptor->PortDescriptors);
    for (lIndex = 0; lIndex < psDescriptor->PortCount; lIndex++)
      free((char *)(psDescriptor->PortNames[lIndex]));
    free((char **)psDescriptor->PortNames);
    free((LADSPA_PortRangeHint *)psDescriptor->PortRangeHints);
    free(psDescriptor);
  }
}

/*****************************************************************************/

/* _fini() is called automatically when the library is unloaded. */
void
_fini() {
  deleteDescriptor(g_psMonoDescriptor);
  deleteDescriptor(g_psStereoDescriptor);
}

/*****************************************************************************/

/* Return a descriptor of the requested plugin type. There are two
   plugin types available in this library (mono and stereo). */
const LADSPA_Descriptor * 
ladspa_descriptor(unsigned long Index) {
  /* Return the requested descriptor or null if the index is out of
     range. */
  switch (Index) {
  case 0:
    return g_psMonoDescriptor;
  case 1:
    return g_psStereoDescriptor;
  default:
    return NULL;
  }
}

/*****************************************************************************/

/* EOF */
